# ================================================================
# SmartCashBackEvo - Configuration Script
# ================================================================
# This script helps configure the SmartCashBackEvo application
# ================================================================

#Requires -RunAsAdministrator

param(
    [string]$InstallPath = "C:\SmartCashBackEvo",
    [string]$ServiceName = "SmartCashBackEvo"
)

$ErrorActionPreference = "Stop"

Write-Host "===============================================" -ForegroundColor Cyan
Write-Host "SmartCashBackEvo - Configuration" -ForegroundColor Cyan
Write-Host "===============================================" -ForegroundColor Cyan
Write-Host ""

$AppSettingsPath = Join-Path $InstallPath "app\appsettings.Production.json"

if (-not (Test-Path $AppSettingsPath)) {
    Write-Host "ERROR: Configuration file not found: $AppSettingsPath" -ForegroundColor Red
    exit 1
}

# Load current settings
$appSettings = Get-Content $AppSettingsPath -Raw | ConvertFrom-Json

Write-Host "Current configuration:" -ForegroundColor Yellow
Write-Host "  HTTP URL: $($appSettings.Kestrel.Endpoints.Http.Url)" -ForegroundColor Gray
if ($appSettings.Kestrel.Endpoints.Https) {
    Write-Host "  HTTPS URL: $($appSettings.Kestrel.Endpoints.Https.Url)" -ForegroundColor Gray
}
Write-Host "  Client URL: $($appSettings.ClientUrl)" -ForegroundColor Gray
Write-Host "  Database: $($appSettings.ConnectionStrings.DefaultConnection)" -ForegroundColor Gray
Write-Host ""

Write-Host "Select configuration option:" -ForegroundColor Yellow
Write-Host "  1. Change HTTP/HTTPS ports" -ForegroundColor Gray
Write-Host "  2. Change Client URL" -ForegroundColor Gray
Write-Host "  3. Configure logging level" -ForegroundColor Gray
Write-Host "  4. View current configuration" -ForegroundColor Gray
Write-Host "  5. Advanced: Edit JSON manually" -ForegroundColor Gray
Write-Host "  0. Exit" -ForegroundColor Gray
Write-Host ""

$choice = Read-Host "Enter your choice"

switch ($choice) {
    "1" {
        # Change ports
        Write-Host ""
        Write-Host "Change HTTP/HTTPS ports:" -ForegroundColor Yellow

        $currentHttpPort = $appSettings.Kestrel.Endpoints.Http.Url -replace '.*:', ''
        $newHttpPort = Read-Host "Enter new HTTP port (current: $currentHttpPort, press Enter to keep)"
        if ([string]::IsNullOrWhiteSpace($newHttpPort)) {
            $newHttpPort = $currentHttpPort
        }

        if ($appSettings.Kestrel.Endpoints.Https) {
            $currentHttpsPort = $appSettings.Kestrel.Endpoints.Https.Url -replace '.*:', ''
            $newHttpsPort = Read-Host "Enter new HTTPS port (current: $currentHttpsPort, press Enter to keep)"
            if ([string]::IsNullOrWhiteSpace($newHttpsPort)) {
                $newHttpsPort = $currentHttpsPort
            }

            # Update HTTPS port
            $appSettings.Kestrel.Endpoints.Https.Url = "https://localhost:$newHttpsPort"

            # Update firewall rule
            Write-Host "Updating firewall rules..." -ForegroundColor Yellow
            Remove-NetFirewallRule -DisplayName "SmartCashBackEvo HTTPS" -ErrorAction SilentlyContinue
            New-NetFirewallRule -DisplayName "SmartCashBackEvo HTTPS" `
                -Direction Inbound `
                -Protocol TCP `
                -LocalPort $newHttpsPort `
                -Action Allow `
                -Profile Any `
                -Enabled True | Out-Null

            # Update Client URL
            $appSettings.ClientUrl = "https://localhost:$newHttpsPort"
        }

        # Update HTTP port
        $appSettings.Kestrel.Endpoints.Http.Url = "http://localhost:$newHttpPort"

        # Update firewall rule
        Remove-NetFirewallRule -DisplayName "SmartCashBackEvo HTTP" -ErrorAction SilentlyContinue
        New-NetFirewallRule -DisplayName "SmartCashBackEvo HTTP" `
            -Direction Inbound `
            -Protocol TCP `
            -LocalPort $newHttpPort `
            -Action Allow `
            -Profile Any `
            -Enabled True | Out-Null

        Write-Host "Ports updated successfully!" -ForegroundColor Green
    }

    "2" {
        # Change Client URL
        Write-Host ""
        Write-Host "Change Client URL:" -ForegroundColor Yellow
        Write-Host "Current: $($appSettings.ClientUrl)" -ForegroundColor Gray
        Write-Host ""
        Write-Host "Examples:" -ForegroundColor Gray
        Write-Host "  https://localhost:5001" -ForegroundColor Gray
        Write-Host "  https://cashback.example.com" -ForegroundColor Gray
        Write-Host "  https://192.168.1.100:5001" -ForegroundColor Gray
        Write-Host ""

        $newClientUrl = Read-Host "Enter new Client URL"
        if (-not [string]::IsNullOrWhiteSpace($newClientUrl)) {
            $appSettings.ClientUrl = $newClientUrl
            Write-Host "Client URL updated successfully!" -ForegroundColor Green
        }
    }

    "3" {
        # Configure logging
        Write-Host ""
        Write-Host "Configure logging level:" -ForegroundColor Yellow
        Write-Host "Current levels:" -ForegroundColor Gray
        $appSettings.Logging.LogLevel | Get-Member -MemberType NoteProperty | ForEach-Object {
            $name = $_.Name
            $value = $appSettings.Logging.LogLevel.$name
            Write-Host "  $name : $value" -ForegroundColor Gray
        }
        Write-Host ""
        Write-Host "Available levels: Trace, Debug, Information, Warning, Error, Critical, None" -ForegroundColor Gray
        Write-Host ""

        $logCategory = Read-Host "Enter category to change (Default, Microsoft.AspNetCore, etc.)"
        $logLevel = Read-Host "Enter new log level"

        if (-not [string]::IsNullOrWhiteSpace($logCategory) -and -not [string]::IsNullOrWhiteSpace($logLevel)) {
            if (-not $appSettings.Logging.LogLevel.$logCategory) {
                $appSettings.Logging.LogLevel | Add-Member -MemberType NoteProperty -Name $logCategory -Value $logLevel
            } else {
                $appSettings.Logging.LogLevel.$logCategory = $logLevel
            }
            Write-Host "Logging level updated successfully!" -ForegroundColor Green
        }
    }

    "4" {
        # View current configuration
        Write-Host ""
        Write-Host "Current configuration (appsettings.Production.json):" -ForegroundColor Yellow
        Write-Host ""
        $appSettings | ConvertTo-Json -Depth 10 | Write-Host
        Write-Host ""
        Read-Host "Press Enter to continue"
        & $MyInvocation.MyCommand.Path -InstallPath $InstallPath -ServiceName $ServiceName
        exit 0
    }

    "5" {
        # Edit manually
        Write-Host ""
        Write-Host "Opening configuration file in notepad..." -ForegroundColor Yellow
        notepad $AppSettingsPath
        Write-Host "Please save and close notepad when done." -ForegroundColor Gray
        Read-Host "Press Enter when ready to continue"
        Write-Host ""
        Write-Host "Configuration file can be manually edited at:" -ForegroundColor Yellow
        Write-Host "  $AppSettingsPath" -ForegroundColor Cyan
        exit 0
    }

    "0" {
        Write-Host "Exiting configuration." -ForegroundColor Yellow
        exit 0
    }

    default {
        Write-Host "Invalid choice." -ForegroundColor Red
        exit 1
    }
}

# Save configuration
Write-Host ""
Write-Host "Saving configuration..." -ForegroundColor Yellow
$appSettings | ConvertTo-Json -Depth 10 | Set-Content $AppSettingsPath -Encoding UTF8
Write-Host "Configuration saved successfully!" -ForegroundColor Green

# Ask to restart service
$service = Get-Service -Name $ServiceName -ErrorAction SilentlyContinue
if ($service) {
    Write-Host ""
    $restart = Read-Host "Do you want to restart the service to apply changes? (y/n)"
    if ($restart -eq 'y') {
        Write-Host "Restarting service..." -ForegroundColor Yellow
        Restart-Service -Name $ServiceName -Force
        Start-Sleep -Seconds 3

        $service = Get-Service -Name $ServiceName
        if ($service.Status -eq 'Running') {
            Write-Host "Service restarted successfully!" -ForegroundColor Green
        } else {
            Write-Host "WARNING: Service failed to start. Check logs for details." -ForegroundColor Yellow
        }
    } else {
        Write-Host "Please restart the service manually to apply changes:" -ForegroundColor Yellow
        Write-Host "  Restart-Service $ServiceName" -ForegroundColor Cyan
    }
}

Write-Host ""
Write-Host "Configuration completed!" -ForegroundColor Green
Write-Host ""
