# ================================================================
# SmartCashBackEvo - Install Windows Service Script
# ================================================================
# This script installs SmartCashBackEvo as a Windows Service
# ================================================================

#Requires -RunAsAdministrator

param(
    [string]$ServiceName = "SmartCashBackEvo",
    [string]$DisplayName = "Smart CashBack Evo Service",
    [string]$Description = "SmartCashBackEvo - Refund Management System",
    [string]$InstallPath = "C:\SmartCashBackEvo",
    [string]$HttpPort = "5000",
    [string]$HttpsPort = "5001",
    [switch]$StartService
)

$ErrorActionPreference = "Stop"

Write-Host "===============================================" -ForegroundColor Cyan
Write-Host "SmartCashBackEvo - Service Installation" -ForegroundColor Cyan
Write-Host "===============================================" -ForegroundColor Cyan
Write-Host ""

# Check if running as Administrator
$currentPrincipal = New-Object Security.Principal.WindowsPrincipal([Security.Principal.WindowsIdentity]::GetCurrent())
if (-not $currentPrincipal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {
    Write-Host "ERROR: This script must be run as Administrator!" -ForegroundColor Red
    exit 1
}

# Get current script location
$ScriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$DistributionRoot = Split-Path -Parent $ScriptDir

Write-Host "Installation parameters:" -ForegroundColor Yellow
Write-Host "  Service Name: $ServiceName" -ForegroundColor Gray
Write-Host "  Display Name: $DisplayName" -ForegroundColor Gray
Write-Host "  Install Path: $InstallPath" -ForegroundColor Gray
Write-Host "  HTTP Port: $HttpPort" -ForegroundColor Gray
Write-Host "  HTTPS Port: $HttpsPort" -ForegroundColor Gray
Write-Host ""

# Check if service already exists
$existingService = Get-Service -Name $ServiceName -ErrorAction SilentlyContinue
if ($existingService) {
    Write-Host "WARNING: Service '$ServiceName' already exists!" -ForegroundColor Yellow
    $response = Read-Host "Do you want to uninstall and reinstall? (y/n)"
    if ($response -ne 'y') {
        Write-Host "Installation cancelled." -ForegroundColor Yellow
        exit 0
    }

    Write-Host "Stopping and removing existing service..." -ForegroundColor Yellow
    Stop-Service -Name $ServiceName -Force -ErrorAction SilentlyContinue
    & sc.exe delete $ServiceName
    Start-Sleep -Seconds 2
}

# Create installation directory
Write-Host "Creating installation directory..." -ForegroundColor Yellow
if (-not (Test-Path $InstallPath)) {
    New-Item -ItemType Directory -Path $InstallPath -Force | Out-Null
}

# Copy application files
Write-Host "Copying application files..." -ForegroundColor Yellow
$AppSource = Join-Path $DistributionRoot "app"
$AppDest = Join-Path $InstallPath "app"

if (Test-Path $AppDest) {
    Write-Host "  Removing old application files..." -ForegroundColor Gray
    Remove-Item -Path $AppDest -Recurse -Force
}

Copy-Item -Path $AppSource -Destination $AppDest -Recurse -Force
Write-Host "  Application files copied successfully" -ForegroundColor Green

# Create data directory
Write-Host "Creating data directory..." -ForegroundColor Yellow
$DataPath = Join-Path $InstallPath "data"
if (-not (Test-Path $DataPath)) {
    New-Item -ItemType Directory -Path $DataPath -Force | Out-Null
}

# Create logs directory
Write-Host "Creating logs directory..." -ForegroundColor Yellow
$LogsPath = Join-Path $InstallPath "logs"
if (-not (Test-Path $LogsPath)) {
    New-Item -ItemType Directory -Path $LogsPath -Force | Out-Null
}

# Update appsettings.Production.json
Write-Host "Configuring application settings..." -ForegroundColor Yellow
$AppSettingsPath = Join-Path $AppDest "appsettings.Production.json"

if (Test-Path $AppSettingsPath) {
    $appSettings = Get-Content $AppSettingsPath -Raw | ConvertFrom-Json

    # Update connection strings to use data directory
    $appSettings.ConnectionStrings.DefaultConnection = "Data Source=$DataPath\app.db"
    $appSettings.ConnectionStrings.RefundConnection = "Data Source=$DataPath\RefundDB.db"

    # Update URLs
    $appSettings.ClientUrl = "https://localhost:$HttpsPort"

    # Update Kestrel configuration
    if (-not $appSettings.Kestrel) {
        $appSettings | Add-Member -MemberType NoteProperty -Name "Kestrel" -Value @{}
    }

    $appSettings.Kestrel = @{
        Endpoints = @{
            Http = @{
                Url = "http://localhost:$HttpPort"
            }
            Https = @{
                Url = "https://localhost:$HttpsPort"
                Certificate = @{
                    Path = "$InstallPath\certificate.pfx"
                    Password = ""
                }
            }
        }
    }

    $appSettings | ConvertTo-Json -Depth 10 | Set-Content $AppSettingsPath -Encoding UTF8
    Write-Host "  Configuration updated successfully" -ForegroundColor Green
}

# Find the executable
$ExePath = Join-Path $AppDest "SmartCashBackEvo.exe"
if (-not (Test-Path $ExePath)) {
    Write-Host "ERROR: Application executable not found at: $ExePath" -ForegroundColor Red
    exit 1
}

# Set environment variables for the service
$env:ASPNETCORE_ENVIRONMENT = "Production"
$env:ASPNETCORE_URLS = "http://localhost:$HttpPort;https://localhost:$HttpsPort"

# Create Windows Service
Write-Host "Creating Windows Service..." -ForegroundColor Yellow
& sc.exe create $ServiceName `
    binPath= "`"$ExePath`"" `
    DisplayName= $DisplayName `
    start= auto

if ($LASTEXITCODE -ne 0) {
    Write-Host "ERROR: Failed to create Windows Service!" -ForegroundColor Red
    exit 1
}

# Set service description
& sc.exe description $ServiceName $Description

# Configure service recovery options (restart on failure)
Write-Host "Configuring service recovery options..." -ForegroundColor Yellow
& sc.exe failure $ServiceName reset= 86400 actions= restart/60000/restart/60000/restart/60000

# Set service to run under NetworkService account (more secure than LocalSystem)
Write-Host "Configuring service account..." -ForegroundColor Yellow
& sc.exe config $ServiceName obj= "NT AUTHORITY\NetworkService"

# Set environment variables for the service (VERSION from version.txt if exists)
Write-Host "Configuring service environment variables..." -ForegroundColor Yellow
$versionFilePath = Join-Path $AppPath "version.txt"
$version = "dev"
if (Test-Path $versionFilePath) {
    $version = (Get-Content $versionFilePath -Raw).Trim()
}

# Create registry key for service environment variables
$servicePath = "HKLM:\SYSTEM\CurrentControlSet\Services\$ServiceName"
New-ItemProperty -Path $servicePath -Name "Environment" -Value @(
    "ASPNETCORE_ENVIRONMENT=Production",
    "ASPNETCORE_URLS=http://localhost:$HttpPort;https://localhost:$HttpsPort",
    "VERSION=$version"
) -PropertyType MultiString -Force | Out-Null

# Grant permissions to data and logs directories
Write-Host "Configuring directory permissions..." -ForegroundColor Yellow
$acl = Get-Acl $DataPath
$permission = "NT AUTHORITY\NetworkService", "FullControl", "ContainerInherit,ObjectInherit", "None", "Allow"
$accessRule = New-Object System.Security.AccessControl.FileSystemAccessRule $permission
$acl.SetAccessRule($accessRule)
Set-Acl $DataPath $acl

$acl = Get-Acl $LogsPath
$acl.SetAccessRule($accessRule)
Set-Acl $LogsPath $acl

Write-Host ""
Write-Host "===============================================" -ForegroundColor Green
Write-Host "Service installed successfully!" -ForegroundColor Green
Write-Host "===============================================" -ForegroundColor Green
Write-Host ""

Write-Host "Next steps:" -ForegroundColor Yellow
Write-Host "  1. Configure SSL certificate (run setup-ssl.ps1)" -ForegroundColor Gray
Write-Host "  2. Review and update appsettings.Production.json if needed" -ForegroundColor Gray
Write-Host "  3. Configure firewall rules for ports $HttpPort and $HttpsPort" -ForegroundColor Gray
Write-Host "  4. Start the service:" -ForegroundColor Gray
Write-Host "     Start-Service $ServiceName" -ForegroundColor Cyan
Write-Host ""

if ($StartService) {
    Write-Host "Starting service..." -ForegroundColor Yellow
    Start-Service -Name $ServiceName
    Start-Sleep -Seconds 3

    $service = Get-Service -Name $ServiceName
    if ($service.Status -eq 'Running') {
        Write-Host "Service started successfully!" -ForegroundColor Green
        Write-Host "Application URL: https://localhost:$HttpsPort" -ForegroundColor Cyan
    } else {
        Write-Host "WARNING: Service failed to start. Check logs for details." -ForegroundColor Yellow
        Write-Host "Logs location: $LogsPath" -ForegroundColor Gray
    }
}

Write-Host ""
Write-Host "Installation completed!" -ForegroundColor Green
Write-Host ""
