# ================================================================
# SmartCashBackEvo - SSL Certificate Setup Script
# ================================================================
# This script helps configure SSL certificate for the application
# Options:
#   1. Generate self-signed certificate
#   2. Use existing certificate
# ================================================================

#Requires -RunAsAdministrator

param(
    [string]$InstallPath = "C:\SmartCashBackEvo",
    [string]$CertificatePassword = "",
    [string]$DomainName = "localhost",
    [int]$ValidityDays = 3650
)

$ErrorActionPreference = "Stop"

Write-Host "===============================================" -ForegroundColor Cyan
Write-Host "SmartCashBackEvo - SSL Certificate Setup" -ForegroundColor Cyan
Write-Host "===============================================" -ForegroundColor Cyan
Write-Host ""

# Check if running as Administrator
$currentPrincipal = New-Object Security.Principal.WindowsPrincipal([Security.Principal.WindowsIdentity]::GetCurrent())
if (-not $currentPrincipal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {
    Write-Host "ERROR: This script must be run as Administrator!" -ForegroundColor Red
    exit 1
}

# Certificate paths
$CertPath = Join-Path $InstallPath "certificate.pfx"
$AppSettingsPath = Join-Path $InstallPath "app\appsettings.Production.json"

Write-Host "Select SSL certificate option:" -ForegroundColor Yellow
Write-Host "  1. Generate self-signed certificate (recommended for testing)" -ForegroundColor Gray
Write-Host "  2. Use existing certificate (.pfx file)" -ForegroundColor Gray
Write-Host "  3. Skip SSL configuration (HTTP only)" -ForegroundColor Gray
Write-Host ""

$choice = Read-Host "Enter your choice (1, 2, or 3)"

switch ($choice) {
    "1" {
        # Generate self-signed certificate
        Write-Host ""
        Write-Host "Generating self-signed certificate..." -ForegroundColor Yellow

        # Ask for certificate details
        if (-not $DomainName) {
            $DomainName = Read-Host "Enter domain name (or press Enter for 'localhost')"
            if ([string]::IsNullOrWhiteSpace($DomainName)) {
                $DomainName = "localhost"
            }
        }

        # Ask for password if not provided
        if ([string]::IsNullOrWhiteSpace($CertificatePassword)) {
            Write-Host "Enter password for certificate (or press Enter for no password):" -ForegroundColor Yellow
            $securePassword = Read-Host -AsSecureString
            $CertificatePassword = [Runtime.InteropServices.Marshal]::PtrToStringAuto(
                [Runtime.InteropServices.Marshal]::SecureStringToBSTR($securePassword))
        }

        Write-Host "  Domain: $DomainName" -ForegroundColor Gray
        Write-Host "  Validity: $ValidityDays days (~$([math]::Round($ValidityDays/365, 1)) years)" -ForegroundColor Gray
        Write-Host ""

        # Create certificate
        $certParams = @{
            Subject = "CN=$DomainName"
            DnsName = @($DomainName, "localhost", "127.0.0.1")
            KeyAlgorithm = "RSA"
            KeyLength = 2048
            HashAlgorithm = "SHA256"
            NotAfter = (Get-Date).AddDays($ValidityDays)
            CertStoreLocation = "Cert:\LocalMachine\My"
            KeyExportPolicy = "Exportable"
            KeyUsage = "DigitalSignature", "KeyEncipherment"
            TextExtension = @("2.5.29.37={text}1.3.6.1.5.5.7.3.1") # Server Authentication
        }

        Write-Host "Creating certificate..." -ForegroundColor Yellow
        $cert = New-SelfSignedCertificate @certParams

        # Export to PFX
        Write-Host "Exporting certificate to PFX..." -ForegroundColor Yellow
        if ([string]::IsNullOrWhiteSpace($CertificatePassword)) {
            Export-PfxCertificate -Cert $cert -FilePath $CertPath -Password (New-Object System.Security.SecureString)
        } else {
            $securePwd = ConvertTo-SecureString -String $CertificatePassword -Force -AsPlainText
            Export-PfxCertificate -Cert $cert -FilePath $CertPath -Password $securePwd
        }

        Write-Host "Certificate generated successfully!" -ForegroundColor Green
        Write-Host "  Location: $CertPath" -ForegroundColor Gray
        Write-Host "  Thumbprint: $($cert.Thumbprint)" -ForegroundColor Gray
        Write-Host ""

        # Trust the certificate (add to Trusted Root)
        Write-Host "Adding certificate to Trusted Root Certification Authorities..." -ForegroundColor Yellow
        $store = New-Object System.Security.Cryptography.X509Certificates.X509Store(
            "Root", "LocalMachine")
        $store.Open("ReadWrite")
        $store.Add($cert)
        $store.Close()

        Write-Host "Certificate trusted successfully!" -ForegroundColor Green
        Write-Host ""
    }

    "2" {
        # Use existing certificate
        Write-Host ""
        Write-Host "Using existing certificate..." -ForegroundColor Yellow

        $existingCertPath = Read-Host "Enter path to existing .pfx certificate file"

        if (-not (Test-Path $existingCertPath)) {
            Write-Host "ERROR: Certificate file not found: $existingCertPath" -ForegroundColor Red
            exit 1
        }

        # Ask for password
        Write-Host "Enter certificate password (or press Enter if no password):" -ForegroundColor Yellow
        $securePassword = Read-Host -AsSecureString
        $CertificatePassword = [Runtime.InteropServices.Marshal]::PtrToStringAuto(
            [Runtime.InteropServices.Marshal]::SecureStringToBSTR($securePassword))

        # Verify certificate can be loaded
        try {
            if ([string]::IsNullOrWhiteSpace($CertificatePassword)) {
                $testCert = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2($existingCertPath)
            } else {
                $securePwd = ConvertTo-SecureString -String $CertificatePassword -Force -AsPlainText
                $testCert = New-Object System.Security.Cryptography.X509Certificates.X509Certificate2($existingCertPath, $securePwd)
            }

            Write-Host "Certificate loaded successfully!" -ForegroundColor Green
            Write-Host "  Subject: $($testCert.Subject)" -ForegroundColor Gray
            Write-Host "  Issuer: $($testCert.Issuer)" -ForegroundColor Gray
            Write-Host "  Valid Until: $($testCert.NotAfter)" -ForegroundColor Gray
            Write-Host "  Thumbprint: $($testCert.Thumbprint)" -ForegroundColor Gray
            Write-Host ""
        } catch {
            Write-Host "ERROR: Failed to load certificate. Check file and password." -ForegroundColor Red
            Write-Host "Error: $($_.Exception.Message)" -ForegroundColor Red
            exit 1
        }

        # Copy certificate to install location
        Write-Host "Copying certificate..." -ForegroundColor Yellow
        Copy-Item -Path $existingCertPath -Destination $CertPath -Force
        Write-Host "Certificate copied successfully!" -ForegroundColor Green
        Write-Host ""
    }

    "3" {
        # Skip SSL
        Write-Host ""
        Write-Host "Skipping SSL configuration..." -ForegroundColor Yellow
        Write-Host "WARNING: The application will run in HTTP-only mode!" -ForegroundColor Red
        Write-Host "This is NOT recommended for production use." -ForegroundColor Red

        # Update appsettings to disable HTTPS
        if (Test-Path $AppSettingsPath) {
            $appSettings = Get-Content $AppSettingsPath -Raw | ConvertFrom-Json

            # Remove HTTPS endpoint
            if ($appSettings.Kestrel -and $appSettings.Kestrel.Endpoints) {
                $appSettings.Kestrel.Endpoints.PSObject.Properties.Remove('Https')
            }

            # Update ClientUrl to HTTP
            $httpPort = $appSettings.Kestrel.Endpoints.Http.Url -replace '.*:', ''
            $appSettings.ClientUrl = "http://localhost:$httpPort"

            $appSettings | ConvertTo-Json -Depth 10 | Set-Content $AppSettingsPath -Encoding UTF8
            Write-Host "Application configured for HTTP-only mode." -ForegroundColor Yellow
        }

        Write-Host ""
        exit 0
    }

    default {
        Write-Host "Invalid choice. Exiting." -ForegroundColor Red
        exit 1
    }
}

# Update appsettings.Production.json with certificate configuration
Write-Host "Updating application configuration..." -ForegroundColor Yellow

if (Test-Path $AppSettingsPath) {
    $appSettings = Get-Content $AppSettingsPath -Raw | ConvertFrom-Json

    # Update certificate configuration
    if ($appSettings.Kestrel -and $appSettings.Kestrel.Endpoints -and $appSettings.Kestrel.Endpoints.Https) {
        $appSettings.Kestrel.Endpoints.Https.Certificate.Path = $CertPath
        $appSettings.Kestrel.Endpoints.Https.Certificate.Password = $CertificatePassword

        $appSettings | ConvertTo-Json -Depth 10 | Set-Content $AppSettingsPath -Encoding UTF8
        Write-Host "Configuration updated successfully!" -ForegroundColor Green
    }
} else {
    Write-Host "WARNING: appsettings.Production.json not found at: $AppSettingsPath" -ForegroundColor Yellow
}

# Configure firewall rules
Write-Host ""
Write-Host "Configuring Windows Firewall..." -ForegroundColor Yellow

$httpsPort = 5001
if (Test-Path $AppSettingsPath) {
    $appSettings = Get-Content $AppSettingsPath -Raw | ConvertFrom-Json
    if ($appSettings.Kestrel.Endpoints.Https.Url) {
        $httpsPort = $appSettings.Kestrel.Endpoints.Https.Url -replace '.*:', ''
    }
}

# Remove existing rule if exists
Remove-NetFirewallRule -DisplayName "SmartCashBackEvo HTTPS" -ErrorAction SilentlyContinue

# Add new rule
New-NetFirewallRule -DisplayName "SmartCashBackEvo HTTPS" `
    -Direction Inbound `
    -Protocol TCP `
    -LocalPort $httpsPort `
    -Action Allow `
    -Profile Any `
    -Enabled True | Out-Null

Write-Host "Firewall rule created for port $httpsPort" -ForegroundColor Green

Write-Host ""
Write-Host "===============================================" -ForegroundColor Green
Write-Host "SSL Certificate setup completed!" -ForegroundColor Green
Write-Host "===============================================" -ForegroundColor Green
Write-Host ""

if ($choice -eq "1") {
    Write-Host "IMPORTANT: Self-signed certificate notes" -ForegroundColor Yellow
    Write-Host "  - The certificate has been added to Trusted Root" -ForegroundColor Gray
    Write-Host "  - Browsers on this machine will trust it" -ForegroundColor Gray
    Write-Host "  - Other machines will show security warnings" -ForegroundColor Gray
    Write-Host "  - For production, use a certificate from a trusted CA" -ForegroundColor Gray
    Write-Host ""
}

Write-Host "Next steps:" -ForegroundColor Yellow
Write-Host "  1. Review appsettings.Production.json" -ForegroundColor Gray
Write-Host "  2. Start or restart the SmartCashBackEvo service" -ForegroundColor Gray
Write-Host "  3. Test access at https://localhost:$httpsPort" -ForegroundColor Gray
Write-Host ""
