# ================================================================
# SmartCashBackEvo - Uninstall Windows Service Script
# ================================================================
# This script uninstalls SmartCashBackEvo Windows Service
# ================================================================

#Requires -RunAsAdministrator

param(
    [string]$ServiceName = "SmartCashBackEvo",
    [string]$InstallPath = "C:\SmartCashBackEvo",
    [switch]$RemoveData,
    [switch]$Force
)

$ErrorActionPreference = "Stop"

Write-Host "===============================================" -ForegroundColor Cyan
Write-Host "SmartCashBackEvo - Service Uninstallation" -ForegroundColor Cyan
Write-Host "===============================================" -ForegroundColor Cyan
Write-Host ""

# Check if running as Administrator
$currentPrincipal = New-Object Security.Principal.WindowsPrincipal([Security.Principal.WindowsIdentity]::GetCurrent())
if (-not $currentPrincipal.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {
    Write-Host "ERROR: This script must be run as Administrator!" -ForegroundColor Red
    exit 1
}

# Check if service exists
$service = Get-Service -Name $ServiceName -ErrorAction SilentlyContinue
if (-not $service) {
    Write-Host "Service '$ServiceName' not found." -ForegroundColor Yellow

    if (Test-Path $InstallPath) {
        Write-Host "However, installation directory exists at: $InstallPath" -ForegroundColor Yellow
        $removeFiles = Read-Host "Do you want to remove it? (y/n)"
        if ($removeFiles -eq 'y') {
            Write-Host "Removing installation directory..." -ForegroundColor Yellow
            Remove-Item -Path $InstallPath -Recurse -Force
            Write-Host "Installation directory removed." -ForegroundColor Green
        }
    }

    exit 0
}

Write-Host "Service found: $ServiceName" -ForegroundColor Yellow
Write-Host "  Status: $($service.Status)" -ForegroundColor Gray
Write-Host "  Install Path: $InstallPath" -ForegroundColor Gray
Write-Host ""

if (-not $Force) {
    Write-Host "WARNING: This will remove the SmartCashBackEvo service!" -ForegroundColor Red
    $confirm = Read-Host "Are you sure you want to continue? (y/n)"
    if ($confirm -ne 'y') {
        Write-Host "Uninstallation cancelled." -ForegroundColor Yellow
        exit 0
    }
}

# Stop service if running
Write-Host "Stopping service..." -ForegroundColor Yellow
if ($service.Status -eq 'Running') {
    Stop-Service -Name $ServiceName -Force
    Start-Sleep -Seconds 3
    Write-Host "  Service stopped" -ForegroundColor Green
} else {
    Write-Host "  Service already stopped" -ForegroundColor Gray
}

# Delete service
Write-Host "Removing service..." -ForegroundColor Yellow
& sc.exe delete $ServiceName

if ($LASTEXITCODE -eq 0) {
    Write-Host "  Service removed successfully" -ForegroundColor Green
} else {
    Write-Host "  WARNING: Failed to remove service (error code: $LASTEXITCODE)" -ForegroundColor Yellow
}

Start-Sleep -Seconds 2

# Remove firewall rules
Write-Host "Removing firewall rules..." -ForegroundColor Yellow
Remove-NetFirewallRule -DisplayName "SmartCashBackEvo HTTP" -ErrorAction SilentlyContinue
Remove-NetFirewallRule -DisplayName "SmartCashBackEvo HTTPS" -ErrorAction SilentlyContinue
Write-Host "  Firewall rules removed" -ForegroundColor Green

# Ask about data removal
if (Test-Path $InstallPath) {
    Write-Host ""
    Write-Host "Installation directory: $InstallPath" -ForegroundColor Yellow

    if (-not $RemoveData -and -not $Force) {
        $removeFiles = Read-Host "Do you want to remove application files? (y/n)"
        $RemoveData = ($removeFiles -eq 'y')
    }

    if ($RemoveData) {
        $dataPath = Join-Path $InstallPath "data"
        if (Test-Path $dataPath) {
            Write-Host ""
            Write-Host "WARNING: Data directory contains databases!" -ForegroundColor Red
            Write-Host "  Location: $dataPath" -ForegroundColor Gray

            if (-not $Force) {
                $removeDb = Read-Host "Do you want to remove databases? (THIS CANNOT BE UNDONE) (y/n)"
                if ($removeDb -ne 'y') {
                    Write-Host "Backing up data directory..." -ForegroundColor Yellow
                    $backupPath = "$InstallPath`_backup_$(Get-Date -Format 'yyyyMMdd_HHmmss')\data"
                    New-Item -ItemType Directory -Path (Split-Path $backupPath -Parent) -Force | Out-Null
                    Copy-Item -Path $dataPath -Destination $backupPath -Recurse -Force
                    Write-Host "  Data backed up to: $backupPath" -ForegroundColor Green
                }
            }
        }

        Write-Host "Removing installation directory..." -ForegroundColor Yellow
        Remove-Item -Path $InstallPath -Recurse -Force -ErrorAction Continue
        Write-Host "  Installation directory removed" -ForegroundColor Green
    } else {
        Write-Host "Keeping installation files at: $InstallPath" -ForegroundColor Gray
        Write-Host "You can manually delete them later if needed." -ForegroundColor Gray
    }
}

# Remove certificate from Trusted Root (if it's self-signed)
$certPath = Join-Path $InstallPath "certificate.pfx"
if ((Test-Path $certPath) -and -not $RemoveData) {
    Write-Host ""
    Write-Host "Self-signed certificate may still be in Trusted Root store." -ForegroundColor Yellow
    Write-Host "To remove it manually:" -ForegroundColor Gray
    Write-Host "  1. Run: certlm.msc" -ForegroundColor Gray
    Write-Host "  2. Navigate to: Trusted Root Certification Authorities > Certificates" -ForegroundColor Gray
    Write-Host "  3. Find and delete certificates issued to 'localhost' or your domain" -ForegroundColor Gray
}

Write-Host ""
Write-Host "===============================================" -ForegroundColor Green
Write-Host "Uninstallation completed!" -ForegroundColor Green
Write-Host "===============================================" -ForegroundColor Green
Write-Host ""
