﻿#Requires -RunAsAdministrator

<#
.SYNOPSIS
    Скрипт обновления SmartCashBackEvo

.DESCRIPTION
    Автоматическое обновление приложения SmartCashBackEvo с сохранением данных и конфигурации.
    Создает резервные копии перед обновлением.

.PARAMETER UpdatePath
    Путь к новой версии дистрибутива (распакованная папка)

.EXAMPLE
    .\update.ps1 -UpdatePath "C:\Temp\SmartCashBackEvo_v1.4.0"
    .\update.ps1
#>

param(
    [Parameter(Mandatory=$false)]
    [string]$UpdatePath = ""
)

$ErrorActionPreference = "Stop"

# Автоматическое определение пути к дистрибутиву
# Если UpdatePath не указан, используем родительскую папку от текущего скрипта
if ([string]::IsNullOrWhiteSpace($UpdatePath)) {
    $ScriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
    $UpdatePath = Split-Path -Parent $ScriptDir
    Write-Host "Автоматически определен путь к обновлению: $UpdatePath" -ForegroundColor Gray
}

# Константы
$ServiceName = "SmartCashBackEvo"
$InstallPath = "C:\SmartCashBackEvo"
$BackupRootPath = "C:\SmartCashBackEvo_Backups"
$DataDir = Join-Path $InstallPath "data"
$AppDir = Join-Path $InstallPath "app"

# Цвета для вывода
function Write-ColorOutput {
    param(
        [string]$Message,
        [string]$Color = "White"
    )
    Write-Host $Message -ForegroundColor $Color
}

function Write-Step {
    param([string]$Message)
    Write-Host ""
    Write-ColorOutput "===> $Message" "Cyan"
}

function Write-Success {
    param([string]$Message)
    Write-ColorOutput "✓ $Message" "Green"
}

function Write-Warning {
    param([string]$Message)
    Write-ColorOutput "⚠ $Message" "Yellow"
}

function Write-Error {
    param([string]$Message)
    Write-ColorOutput "✗ $Message" "Red"
}

# Проверка прав администратора
if (-not ([Security.Principal.WindowsPrincipal][Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {
    Write-Error "Этот скрипт требует прав администратора!"
    Write-Host "Запустите PowerShell от имени администратора и попробуйте снова."
    exit 1
}

Write-ColorOutput "╔═══════════════════════════════════════════════════════╗" "Cyan"
Write-ColorOutput "║   Обновление SmartCashBackEvo                         ║" "Cyan"
Write-ColorOutput "╚═══════════════════════════════════════════════════════╝" "Cyan"
Write-Host ""

# Запрос пути к обновлению если не указан
if ([string]::IsNullOrWhiteSpace($UpdatePath)) {
    Write-Host "Укажите путь к распакованной папке с новой версией:" -ForegroundColor Yellow
    Write-Host "Пример: C:\Temp\SmartCashBackEvo_Distribution_v1.4.2" -ForegroundColor Gray
    Write-Host ""
    $UpdatePath = Read-Host "Путь к новой версии"

    if ([string]::IsNullOrWhiteSpace($UpdatePath)) {
        Write-Error "Путь к обновлению не указан"
        Write-Host ""
        Write-Host "Использование:" -ForegroundColor Cyan
        Write-Host "  .\update.ps1 -UpdatePath `"C:\Temp\SmartCashBackEvo_v1.4.2`""
        Write-Host "или запустите без параметров и введите путь интерактивно"
        exit 1
    }
}

# 1. Проверка существования установки
Write-Step "Шаг 1: Проверка текущей установки"

if (-not (Test-Path $InstallPath)) {
    Write-Error "SmartCashBackEvo не установлен в $InstallPath"
    Write-Host "Используйте install-service.ps1 для первоначальной установки."
    exit 1
}

if (-not (Get-Service -Name $ServiceName -ErrorAction SilentlyContinue)) {
    Write-Error "Служба $ServiceName не найдена"
    exit 1
}

Write-Success "Текущая установка найдена: $InstallPath"

# 2. Проверка пути к обновлению
Write-Step "Шаг 2: Проверка пакета обновления"

if (-not (Test-Path $UpdatePath)) {
    Write-Error "Путь к обновлению не найден: $UpdatePath"
    exit 1
}

$updateAppPath = Join-Path $UpdatePath "app"
if (-not (Test-Path $updateAppPath)) {
    Write-Error "Папка 'app' не найдена в $UpdatePath"
    Write-Host "Убедитесь, что вы указали путь к распакованному дистрибутиву."
    exit 1
}

$updateExe = Join-Path $updateAppPath "SmartCashBackEvo.exe"
if (-not (Test-Path $updateExe)) {
    Write-Error "SmartCashBackEvo.exe не найден в $updateAppPath"
    exit 1
}

Write-Success "Пакет обновления проверен: $UpdatePath"

# 3. Создание резервной копии
Write-Step "Шаг 3: Создание резервной копии"

$timestamp = Get-Date -Format "yyyy-MM-dd_HH-mm-ss"
$backupPath = Join-Path $BackupRootPath "Backup_$timestamp"
$backupAppPath = Join-Path $backupPath "app"

# Определяем конфигурационные файлы для резервного копирования
$configFiles = @(
    "appsettings.json",
    "appsettings.Production.json",
    "*.pfx"
)

try {
    # Создаем директорию для резервных копий
    if (-not (Test-Path $BackupRootPath)) {
        New-Item -ItemType Directory -Path $BackupRootPath -Force | Out-Null
    }

    New-Item -ItemType Directory -Path $backupPath -Force | Out-Null
    Write-Host "Создана папка резервной копии: $backupPath"

    # Копируем текущее приложение
    Write-Host "Копирование приложения..."
    Copy-Item -Path $AppDir -Destination $backupAppPath -Recurse -Force
    Write-Success "Приложение скопировано"

    # Копируем данные (базы данных)
    if (Test-Path $DataDir) {
        Write-Host "Копирование данных..."
        $backupDataPath = Join-Path $backupPath "data"
        Copy-Item -Path $DataDir -Destination $backupDataPath -Recurse -Force
        Write-Success "Данные скопированы"
    }

    # Копируем конфигурацию
    Write-Host "Копирование конфигурации..."

    foreach ($pattern in $configFiles) {
        $files = Get-ChildItem -Path $AppDir -Filter $pattern -ErrorAction SilentlyContinue
        foreach ($file in $files) {
            Copy-Item -Path $file.FullName -Destination $backupAppPath -Force
        }
    }
    Write-Success "Конфигурация скопирована"

    # Сохраняем информацию о версии
    $versionInfo = @"
Backup created: $timestamp
Original installation path: $InstallPath
Backed up by: $env:USERNAME
Computer: $env:COMPUTERNAME
"@
    $versionInfo | Out-File -FilePath (Join-Path $backupPath "backup-info.txt") -Encoding UTF8

    Write-Success "Резервная копия создана: $backupPath"
} catch {
    Write-Error "Ошибка при создании резервной копии: $_"
    exit 1
}

# 4. Остановка службы
Write-Step "Шаг 4: Остановка службы"

try {
    $service = Get-Service -Name $ServiceName
    if ($service.Status -eq "Running") {
        Write-Host "Остановка службы $ServiceName..."
        Stop-Service -Name $ServiceName -Force

        # Ждем остановки
        $timeout = 30
        $timer = [Diagnostics.Stopwatch]::StartNew()
        while ((Get-Service -Name $ServiceName).Status -ne "Stopped" -and $timer.Elapsed.TotalSeconds -lt $timeout) {
            Start-Sleep -Seconds 1
        }

        if ((Get-Service -Name $ServiceName).Status -eq "Stopped") {
            Write-Success "Служба остановлена"
        } else {
            Write-Warning "Служба не остановилась за $timeout секунд, продолжаем..."
        }
    } else {
        Write-Success "Служба уже остановлена"
    }

    # Дополнительное ожидание и проверка завершения процесса
    Write-Host "Ожидание завершения процессов и освобождения файлов..."
    Start-Sleep -Seconds 5

    # Проверяем, что процесс SmartCashBackEvo.exe действительно завершен
    $processTimeout = 10
    $processTimer = [Diagnostics.Stopwatch]::StartNew()
    while ((Get-Process -Name "SmartCashBackEvo" -ErrorAction SilentlyContinue) -and $processTimer.Elapsed.TotalSeconds -lt $processTimeout) {
        Write-Host "Ожидание завершения процесса SmartCashBackEvo.exe..."
        Start-Sleep -Seconds 1
    }

    if (Get-Process -Name "SmartCashBackEvo" -ErrorAction SilentlyContinue) {
        Write-Warning "Процесс SmartCashBackEvo.exe все еще запущен, принудительное завершение..."
        Stop-Process -Name "SmartCashBackEvo" -Force -ErrorAction SilentlyContinue
        Start-Sleep -Seconds 2
    }

    Write-Success "Процессы завершены"
} catch {
    Write-Error "Ошибка при остановке службы: $_"
    Write-Host "Попробуйте остановить службу вручную и запустите скрипт снова."
    exit 1
}

# 5. Обновление файлов приложения
Write-Step "Шаг 5: Обновление файлов приложения"

try {
    # Сохраняем важные файлы конфигурации
    # Получаем полный путь к временной директории, избегая сокращенных 8.3 имен
    $tempRoot = [System.IO.Path]::GetTempPath()
    if (-not $tempRoot) {
        $tempRoot = $env:TEMP
    }
    # Преобразуем в полный путь если это 8.3 имя
    $tempRoot = (Get-Item $tempRoot).FullName
    $tempConfigPath = Join-Path $tempRoot "SmartCashBackEvo_Config_$timestamp"
    New-Item -ItemType Directory -Path $tempConfigPath -Force | Out-Null

    Write-Host "Сохранение конфигурационных файлов..."
    foreach ($pattern in $configFiles) {
        $files = Get-ChildItem -Path $AppDir -Filter $pattern -ErrorAction SilentlyContinue
        foreach ($file in $files) {
            Copy-Item -Path $file.FullName -Destination $tempConfigPath -Force
        }
    }

    # Удаляем старые файлы приложения (кроме данных)
    Write-Host "Удаление старых файлов приложения..."

    # Повторные попытки удаления файлов (для избежания ошибок блокировки)
    $deleteRetries = 3
    $deleteSuccess = $false

    for ($i = 1; $i -le $deleteRetries; $i++) {
        try {
            Get-ChildItem -Path $AppDir -Exclude "data" | Remove-Item -Recurse -Force -ErrorAction Stop
            $deleteSuccess = $true
            break
        } catch {
            if ($i -lt $deleteRetries) {
                Write-Host "Попытка $i из $deleteRetries не удалась, ожидание 5 секунд..." -ForegroundColor Yellow
                Start-Sleep -Seconds 5
            } else {
                throw $_
            }
        }
    }

    if (-not $deleteSuccess) {
        throw "Не удалось удалить старые файлы после $deleteRetries попыток"
    }

    # Копируем новые файлы приложения
    Write-Host "Копирование новых файлов приложения..."
    Copy-Item -Path "$updateAppPath\*" -Destination $AppDir -Recurse -Force

    # Восстанавливаем конфигурационные файлы
    Write-Host "Восстановление конфигурационных файлов..."
    if (Test-Path $tempConfigPath) {
        $configFilesToRestore = Get-ChildItem -Path $tempConfigPath -ErrorAction SilentlyContinue
        if ($configFilesToRestore) {
            foreach ($file in $configFilesToRestore) {
                Write-Host "  Восстанавливаем: $($file.Name)"
                Copy-Item -Path $file.FullName -Destination $AppDir -Force
            }
        } else {
            Write-Warning "Нет конфигурационных файлов для восстановления"
        }
    } else {
        Write-Warning "Временная папка с конфигурацией не найдена: $tempConfigPath"
    }

    # Удаляем временную папку
    if (Test-Path $tempConfigPath) {
        Remove-Item -Path $tempConfigPath -Recurse -Force -ErrorAction SilentlyContinue
    }

    Write-Success "Файлы приложения обновлены"
} catch {
    Write-Error "Ошибка при обновлении файлов: $_"
    Write-Host ""
    Write-Warning "Для отката к предыдущей версии:"
    Write-Host "1. Скопируйте файлы из $backupAppPath обратно в $AppDir"
    Write-Host "2. Запустите службу: Start-Service $ServiceName"
    exit 1
}

# 6. Проверка и обновление данных
Write-Step "Шаг 6: Проверка структуры данных"

# Убеждаемся, что папка data существует
if (-not (Test-Path $DataDir)) {
    Write-Host "Создание папки данных..."
    New-Item -ItemType Directory -Path $DataDir -Force | Out-Null
}

Write-Success "Структура данных проверена"

# 6.5. Обновление версии в реестре службы
Write-Host "Обновление версии службы..."
$versionFilePath = Join-Path $AppDir "version.txt"
if (Test-Path $versionFilePath) {
    $version = (Get-Content $versionFilePath -Raw).Trim()
    Write-Host "Найдена версия: $version"

    # Обновляем переменную VERSION в реестре службы
    $servicePath = "HKLM:\SYSTEM\CurrentControlSet\Services\$ServiceName"
    try {
        New-ItemProperty -Path $servicePath -Name "Environment" -Value @(
            "ASPNETCORE_ENVIRONMENT=Production",
            "VERSION=$version"
        ) -PropertyType MultiString -Force | Out-Null
        Write-Success "Версия службы обновлена: $version"
    } catch {
        Write-Warning "Не удалось обновить версию в реестре: $_"
    }
} else {
    Write-Warning "Файл version.txt не найден, версия не обновлена"
}

# 7. Запуск службы
Write-Step "Шаг 7: Запуск обновленной службы"

try {
    Write-Host "Запуск службы $ServiceName..."
    Start-Service -Name $ServiceName

    # Ждем запуска
    $timeout = 30
    $timer = [Diagnostics.Stopwatch]::StartNew()
    while ((Get-Service -Name $ServiceName).Status -ne "Running" -and $timer.Elapsed.TotalSeconds -lt $timeout) {
        Start-Sleep -Seconds 1
    }

    if ((Get-Service -Name $ServiceName).Status -eq "Running") {
        Write-Success "Служба запущена успешно"
    } else {
        Write-Warning "Служба не запустилась за $timeout секунд"
        Write-Host "Проверьте логи для диагностики проблемы"
    }
} catch {
    Write-Error "Ошибка при запуске службы: $_"
    Write-Host ""
    Write-Warning "Для отката к предыдущей версии:"
    Write-Host "1. Скопируйте файлы из $backupAppPath обратно в $AppDir"
    Write-Host "2. Запустите службу: Start-Service $ServiceName"
    exit 1
}

# 8. Проверка работоспособности
Write-Step "Шаг 8: Проверка работоспособности"

Start-Sleep -Seconds 5

$service = Get-Service -Name $ServiceName
if ($service.Status -eq "Running") {
    Write-Success "Служба работает корректно"

    # Пробуем подключиться к health endpoint
    try {
        $response = Invoke-WebRequest -Uri "http://localhost:5000/health" -UseBasicParsing -TimeoutSec 5 -ErrorAction Stop
        if ($response.StatusCode -eq 200) {
            Write-Success "Health check пройден успешно"
        }
    } catch {
        Write-Warning "Health check не прошел, но служба работает"
        Write-Host "Возможно, приложение еще запускается. Подождите несколько секунд."
    }
} else {
    Write-Warning "Служба не запущена"
    Write-Host "Статус: $($service.Status)"
}

# 9. Итоговая информация
Write-Host ""
Write-ColorOutput "╔═══════════════════════════════════════════════════════╗" "Green"
Write-ColorOutput "║   Обновление завершено успешно!                       ║" "Green"
Write-ColorOutput "╚═══════════════════════════════════════════════════════╝" "Green"
Write-Host ""

Write-Host "Информация об обновлении:" -ForegroundColor Cyan
Write-Host "  • Путь установки: $InstallPath"
Write-Host "  • Резервная копия: $backupPath"
Write-Host "  • Служба: $ServiceName"
Write-Host ""

Write-Host "Полезные команды:" -ForegroundColor Cyan
Write-Host "  • Проверить статус: Get-Service $ServiceName"
Write-Host "  • Перезапустить: Restart-Service $ServiceName"
Write-Host "  • Остановить: Stop-Service $ServiceName"
Write-Host "  • Посмотреть логи: Get-EventLog -LogName Application -Source $ServiceName -Newest 50"
Write-Host ""

Write-Host "Приложение доступно по адресу:" -ForegroundColor Cyan
Write-Host "  • HTTP: http://localhost:5000"
Write-Host "  • HTTPS: https://localhost:5001 (если настроен SSL)"
Write-Host ""

# Очистка старых резервных копий (старше 30 дней)
Write-Host "Очистка старых резервных копий (старше 30 дней)..." -ForegroundColor Yellow
$oldBackups = Get-ChildItem -Path $BackupRootPath -Directory | Where-Object { $_.CreationTime -lt (Get-Date).AddDays(-30) }
if ($oldBackups) {
    $oldBackups | ForEach-Object {
        Remove-Item -Path $_.FullName -Recurse -Force
        Write-Host "  • Удалена старая резервная копия: $($_.Name)"
    }
} else {
    Write-Host "  • Старых резервных копий не найдено"
}

Write-Host ""
Write-Success "Готово!"
